
#include <math.h>
//#include <assert.h>
#define ASSERT(x)

#include "HUtilityGeometryCreation.h"

#include "hc.h"

/*! 
  GeneratePointOnCircle generates a single point along a circle based on a centerpoint, a radius, and an angle.  
  \param *point A pointer to an HPoint object.
  \param center The centerpoint of the circle.
  \param radius The length of the radius of the circle.
  \param axis XY, XZ, or YZ.  This defines the view plane in which the circle will be drawn, as well as the axis of
  rotation for the angular offset of the point.  XY defines the z-axis, XZ defines the y-axis, and YZ defines the x-axis.
  \param angle Degrees measured from 0 about the axis defined by the 'axis' parameter.
  \return A Boolean value indicating success or failure
*/
bool HUtilityGeometryCreation::GeneratePointOnCircle(HPoint* point, HPoint &center, float radius, int axis, float angle, bool convert)
{
	HPoint temp;

	switch(axis)
	{
	case HUtility::XZ:
		temp.Set(center.x + radius * (float)cos (angle),
				center.y,
				center.z + radius * (float)sin (angle));
		break;
	case HUtility::XY:
		temp.Set(center.x + radius * (float)cos (angle),
				center.y + radius * (float)sin (angle),
				center.z);
		break;
	case HUtility::YZ:
		temp.Set(center.x,
				center.y + radius * (float)cos (angle),
				center.z + radius * (float)sin (angle));
		break;
	default:
		return false;
	}

	if (convert)
	    HC_Compute_Coordinates(".", "viewpoint", &temp, "world", &temp);

 	*point = temp;

	return true;
}

/*! 
  GeneratePointsOnCircle generates multiple points along a circle based a centerpoint and a radius. The desired number of 
  points must be specified, and must be equal to the size of the points array if the size of that array has been defined. 
  Note that the basic work in this function is accomplished by repeated calls to GeneratePointOnCircle.
  \param *points A pointer to an array of HPoint objects.
  \param center The centerpoint of the circle.
  \param radius The radius of the circle.
  \param axis XY, XZ, or YZ. This defines the view plane in which the circle will be drawn.
  \param numpoints Desired number of points in the circle.
  \return A Boolean value indicating success or failure
*/
bool HUtilityGeometryCreation::GeneratePointsOnCircle (HPoint* points, HPoint & center,float radius,int axis, int numpoints, bool convert)
{
    int i;
    bool success = false;
    float angle = 2.0f * HUtility::PI / (float)numpoints;

    for (i=0; i<numpoints; i++)
    {
	success = GeneratePointOnCircle(&points[i], center, radius, axis, angle*i, convert);
    }

    return success;
}


void HUtilityGeometryCreation::CreateSphere(HPoint center, float radius, int numsides, HPoint axis, HPoint ref)
{
	HPoint	hpos;
 	hpos.Set(0,0,0);
 	bool success = false;
	HPoint *points = 0;
	int *faces_list = 0;
	int i;
	int j;

	success=success;

	hpos.y -= radius + radius/(float)numsides;

	int htile, wtile;
	htile = numsides/2;
	wtile = numsides;

	points= new HPoint[htile*wtile];
	ASSERT(points);
	// faces_list will needs wtile + 1 numbers to specify each sphere end cap
	// and 5*numsides numbers for each strip between the end caps.
	// There will be numsides/2-1 of these strips.
	faces_list= new int[(wtile+1)*2 + 5*(htile-1)*wtile];
	ASSERT(faces_list);

	// generate the points for the bottom of the cylinder.

	for (i = 0; i < htile; i++)
	{
		float angle = (.5f+(float)i) * HUtility::PI/(float)htile;
		ASSERT(angle > 0);
		ASSERT(angle < HUtility::PI);
		hpos.y = - radius * (float)cos(angle);	
		success = GeneratePointsOnCircle(points+i*wtile, hpos, radius*(float)sin(angle), HUtility::XZ, wtile, false);
		ASSERT(success);
	}


	// set the number of points in the end caps of the sphere in the connectivity list
	faces_list[5*(htile-1)*wtile] = wtile;
	faces_list[5*(htile-1)*wtile+wtile+1] = wtile;

	for (i = 0; i < (htile-1); i++)
	{
		for (j = 0; j < wtile; j++)
		{
			// generate the connectivity list for the side quads of the cylinder
			faces_list[(i*wtile + j)*5] = 4; // number of points in this polygon
			faces_list[(i*wtile + j)*5+1]=i*wtile + j;
			faces_list[(i*wtile + j)*5+2]=i*wtile + (j+1)%wtile; //next point along circle; wrap at numsides
			faces_list[(i*wtile + j)*5+3]=(i+1)*wtile + (j+1)%wtile;  // correcsponding point at top of cylinder
			faces_list[(i*wtile + j)*5+4]=(i+1)*wtile + j; //next point along circle; wrap at numsides
		}
	}
	for (j = 0; j < wtile; j++)
	{
		// generate the connectivity list for the ends of the cylinder
		faces_list[(htile-1)*wtile*5 + j + 1]= (wtile - j - 1);
		faces_list[(htile-1)*wtile*5 + j + 1 + wtile + 1]= wtile*(htile-1) + j;
	}
	if (points)
	{
 		// put each entity in a separate segment so that we can easily apply
		// transformations to separately created entities	
			HC_Open_Segment("");
				HC_Insert_Shell(htile*wtile, points, (wtile+1)*2 + 5*(htile-1)*wtile, faces_list);
  				HPoint side;
 	 			HC_Compute_Cross_Product(&axis, &ref, &side);
				SetupModellingMatrix(axis, ref, side, center);
   			HC_Close_Segment();
		delete points;
	}
	if (faces_list)
		delete faces_list;
}

void HUtilityGeometryCreation::SetupModellingMatrix(HVector axis, HVector ref, HVector side, HPoint pos)
{
    float mat[16] = {axis.x, axis.y, axis.z, 0, 
			ref.x, ref.y, ref.z, 0,
			side.x, side.y, side.z, 0,
			pos.x, pos.y, pos.z,1};

    HC_Set_Modelling_Matrix(mat);
}



// CreateCone(radius, height, ref_axes1, ref_axes2, num_sides)
void HUtilityGeometryCreation::CreateSphere(const char *segment, HPoint center, float radius, int numsides)
{
    HPoint	hpos;
    
    hpos = center;
    bool success = false;
    HPoint *points = 0;
    int *faces_list = 0;
    int i;
    int j;

	success=success;

    hpos.y -= radius + radius/(float)numsides;

    int htile, wtile;
    htile = numsides/2;
    wtile = numsides;

    points= new HPoint[htile*wtile];
    ASSERT(points);
    // faces_list will needs wtile + 1 numbers to specify each sphere end cap
    // and 5*numsides numbers for each strip between the end caps.
    // There will be numsides/2-1 of these strips.
    faces_list= new int[(wtile+1)*2 + 5*(htile-1)*wtile];
    ASSERT(faces_list);

    // generate the points for the bottom of the cylinder.

    HC_Open_Segment(segment);

	for (i = 0; i < htile; i++)
	{
	    float angle = (.5f+(float)i) * HUtility::PI/(float)htile;
	    ASSERT(angle > 0);
	    ASSERT(angle < HUtility::PI);
	    hpos.y = center.y - radius * (float)cos(angle);	
	    success = GeneratePointsOnCircle(points+i*wtile, hpos, radius*(float)sin(angle), HUtility::XZ, wtile);
	    ASSERT(success);
	}

	// set the number of points in the end caps of the sphere in the connectivity list
	faces_list[5*(htile-1)*wtile] = wtile;
	faces_list[5*(htile-1)*wtile+wtile+1] = wtile;

	for (i = 0; i < (htile-1); i++)
	{
	    for (j = 0; j < wtile; j++)
	    {
		// generate the connectivity list for the side quads of the cylinder
		faces_list[(i*wtile + j)*5] = 4; // number of points in this polygon
		faces_list[(i*wtile + j)*5+1]=i*wtile + j;
		faces_list[(i*wtile + j)*5+2]=i*wtile + (j+1)%wtile; //next point along circle; wrap at numsides
		faces_list[(i*wtile + j)*5+3]=(i+1)*wtile + (j+1)%wtile;  // correcsponding point at top of cylinder
		faces_list[(i*wtile + j)*5+4]=(i+1)*wtile + j; //next point along circle; wrap at numsides
	    }
	}

	for (j = 0; j < wtile; j++)
	{
	    // generate the connectivity list for the ends of the cylinder
	    faces_list[(htile-1)*wtile*5 + j + 1]= (wtile - j - 1);
	    faces_list[(htile-1)*wtile*5 + j + 1 + wtile + 1]= wtile*(htile-1) + j;
	}

	if (points)
	{
	    HC_Insert_Shell(htile*wtile, points, (wtile+1)*2 + 5*(htile-1)*wtile, faces_list);
	    delete points;
	}

    HC_Close_Segment();

    if (faces_list)
	    delete faces_list;
}


void	HUtilityGeometryCreation::CreateCone(HPoint center, float radius, float height, int numsides, HPoint axis, HPoint ref)
{
	HPoint	hpos;
	
	hpos.Set(0,0,0);
	bool success = false;
	HPoint *points = 0;
	int *faces_list = 0;
	int i;
	int j;

	success=success;

	hpos.y -= radius + radius/(float)numsides;

	int htile, wtile;
	htile = numsides/4;
	if (htile < 2) htile = 2;
	wtile = numsides;
	if (wtile < 4) wtile = 4;

	points= new HPoint[(htile+1)*wtile];
	ASSERT(points);
	// faces_list will needs wtile + 1 numbers to specify the cone end cap
	// and 5*numsides numbers for each strip above the end cap.  Multiple
	// strips are used to ensure more accurate gouraud shading.
	// There will be numsides/2-1 of these strips.
	faces_list= new int[(wtile+1) + 5*(htile-1)*wtile];
	ASSERT(faces_list);

	// generate the points for the bottom of the cylinder.

	for (i = 0; i < htile; i++)
	{
//		float factor = (float)pow(2^i);
		float scale = (float)(1.0 - 1.0/pow(2.0, (double)i));
		if (i == (htile - 1)) scale = 1.0f;
		hpos.y = height * scale;	
		success = GeneratePointsOnCircle(points+i*wtile, hpos, radius-radius*scale, HUtility::XZ, wtile, false);
		ASSERT(success);
	}
	hpos.y = 0;	
	success = GeneratePointsOnCircle(points+htile*wtile, hpos, radius, HUtility::XZ, wtile,false);


	// set the number of points in the end caps of the sphere in the connectivity list
	faces_list[5*(htile-1)*wtile] = wtile;

	for (i = 0; i < (htile-1); i++)
	{
		for (j = 0; j < wtile; j++)
		{
			// generate the connectivity list for the side quads of the cylinder
			faces_list[(i*wtile + j)*5] = 4; // number of points in this polygon
			if (height < 0)
			{
				faces_list[(i*wtile + j)*5+2]=i*wtile + j;
				faces_list[(i*wtile + j)*5+1]=i*wtile + (j+1)%wtile; //next point along circle; wrap at numsides
				faces_list[(i*wtile + j)*5+4]=(i+1)*wtile + (j+1)%wtile;  // correcsponding point at top of cylinder
				faces_list[(i*wtile + j)*5+3]=(i+1)*wtile + j; //next point along circle; wrap at numsides
			}
			else
			{
				faces_list[(i*wtile + j)*5+3]=i*wtile + j;
				faces_list[(i*wtile + j)*5+4]=i*wtile + (j+1)%wtile; //next point along circle; wrap at numsides
				faces_list[(i*wtile + j)*5+1]=(i+1)*wtile + (j+1)%wtile;  // correcsponding point at top of cylinder
				faces_list[(i*wtile + j)*5+2]=(i+1)*wtile + j; //next point along circle; wrap at numsides
			}
		}
	}
	for (j = 0; j < wtile; j++)
	{
		// generate the connectivity list for the ends of the cylinder
		if (height < 0)
		{
			faces_list[(htile-1)*wtile*5 + j + 1]= htile*wtile + j;
		}
		else
		{
			faces_list[(htile-1)*wtile*5 + ((wtile-1)-j) + 1]= htile*wtile + j;
		}
	}
	if (points)
	{
		// put each entity in a separate segment so that we can easily apply
		// transformations to separately created entities	
			HC_Open_Segment("");
				HC_Insert_Shell((htile+1)*wtile, points, wtile+1 + 5*(htile-1)*wtile, faces_list);
				HPoint side;
 	 			HC_Compute_Cross_Product(&axis, &ref, &side);
				SetupModellingMatrix(axis,ref,side,center);
			HC_Close_Segment();


		delete points;
	}
	if (faces_list)
		delete faces_list;
}


void	HUtilityGeometryCreation::CreateCylinder(HPoint center, float radius, float height, int numsides, HPoint axis, HPoint ref)
{
	HPoint	bottom, top;
	
	bottom = center;
	bottom.Set(0,0,0);
	bool success = false;
	HPoint *points = 0;
	int *faces_list = 0;
	int i;

	success=success;

	points= new HPoint[numsides*4];
	ASSERT(points);
	// faces_list will needs 5 numbers to specify each side quad
	// and CYLINDER_NUMSIDES+1 numbers to specify each end polygon.
	faces_list= new int[numsides*5 + 2*(numsides+1)];
	ASSERT(faces_list);

	// generate the points for the bottom of the cylinder.

	success = GeneratePointsOnCircle(points, bottom, radius, HUtility::XZ, numsides, false);
	success = GeneratePointsOnCircle(points+numsides*2, bottom, radius, HUtility::XZ, numsides, false);
	ASSERT(success);

	// generate the points for the top of the cylinder
	top.Set(0, height, 0);
	success = GeneratePointsOnCircle(points+numsides, top, radius, HUtility::XZ, numsides, false);
	success = GeneratePointsOnCircle(points+numsides*3, top, radius, HUtility::XZ, numsides, false);
	ASSERT(success);

	// set the number of points in the end caps of the cylinder in the connectivity list
	faces_list[numsides*5] = numsides;
	faces_list[numsides*6+1] = numsides;

	for (i = 0; i < numsides; i++)
	{
		// generate the connectivity list for the side quads of the cylinder
		faces_list[i*5] = 4; // number of points in this polygon
		if (height < 0)
		{
			faces_list[i*5+2]=i;
			faces_list[i*5+1]=(i+1)%numsides; //next point along circle; wrap at numsides
			faces_list[i*5+4]=numsides+((i+1)%numsides);  // correcsponding point at top of cylinder
			faces_list[i*5+3]=numsides+i; //next point along circle; wrap at numsides

			// generate the connectivity list for the ends of the cylinder
			faces_list[numsides*5+i+1]= numsides*2 + i;
			faces_list[numsides*6+i+2]= numsides*3 + (numsides-i-1);
		}
		else
		{
			faces_list[i*5+3]=i;
			faces_list[i*5+4]=(i+1)%numsides; //next point along circle; wrap at numsides
			faces_list[i*5+1]=numsides+((i+1)%numsides);  // correcsponding point at top of cylinder
			faces_list[i*5+2]=numsides+i; //next point along circle; wrap at numsides

			// generate the connectivity list for the ends of the cylinder
			faces_list[numsides*5+i+1]= numsides*2 + (numsides-i-1);
			faces_list[numsides*6+i+2]= numsides*3 + i;
		}
	}

	if (points)
	{
 		// put each entity in a separate segment so that we can easily apply
		// transformations to separately created entities	
			HC_Open_Segment("");
				HC_Insert_Shell(numsides*4, points, numsides*5 + 2*(numsides+1), faces_list);
				HPoint side;
 	 			HC_Compute_Cross_Product(&axis, &ref, &side);
				SetupModellingMatrix(axis,ref,side,center);
  			HC_Close_Segment();
 		delete points;
	}
	if (faces_list)
		delete faces_list;
}


/*!
	CreateCuboid creates a solid box spanning diagonally from min to max point
	NOTE: The difference between this routene and the HUtility::InsertBox is, I 
	have not reused any corner points while inserting the shell so that the cuboid
	appears flat shaded even in Goroud shading mode which is realistic
	\param HPoint *max Max point coord
	\param HPoint *min Min point coord
	\return An integer value indicating success or failure.
*/
void HUtilityGeometryCreation::CreateCuboid( HPoint *max, HPoint *min)
{
    HPoint 	points[24];
    int		face_list[30];

	GenerateCuboidPoints(max, min, points);

	// Face1 in XY
    face_list[0] = 4;
    face_list[1] = 0;
    face_list[2] = 1;
    face_list[3] = 2;
    face_list[4] = 3;

	// Face2 in XY
    face_list[5] = 4;
    face_list[6] = 4;
    face_list[7] = 5;
    face_list[8] = 6;
    face_list[9] = 7;

	// Face1 in YZ
    face_list[10] = 4;
    face_list[11] = 8;
    face_list[12] = 9;
    face_list[13] = 10;
    face_list[14] = 11;

	// Face2 in YZ
    face_list[15] = 4;
    face_list[16] = 12;
    face_list[17] = 13;
    face_list[18] = 14;
    face_list[19] = 15;

	// Face1 in XZ
    face_list[20] = 4;
    face_list[21] = 16;
    face_list[22] = 17;
    face_list[23] = 18;
    face_list[24] = 19;

	// Face2 in XZ
    face_list[25] = 4;
    face_list[26] = 20;
    face_list[27] = 21;
    face_list[28] = 22;
    face_list[29] = 23;

	HC_Insert_Shell (24, points, 30, face_list);
   
    return;
}

/*! 
  GenerateCuboidPoints creates the vertices of a cuboid based on two input points.
  NOTE: I have to generate extra points here for each corner so that the 
  block appear to be flat shaded even in Goroud shading mode which is the only
  realistic appearance for blocks - Rajesh B (16-Feb-01)
  \param *max A pointer to an HPoint object.
  \param *min A pointer to an HPoint object.
  \param *points A pointer to an array of HPoint objects.  Passed by reference.
*/
void HUtilityGeometryCreation::GenerateCuboidPoints( HPoint	*max, HPoint *min, HPoint *points)
{
	// points for Face1 in XY
    points[0].x = min->x;
    points[0].y = min->y;
    points[0].z = min->z;

    points[1].x = max->x;
    points[1].y = min->y;
    points[1].z = min->z;

    points[2].x = max->x;
    points[2].y = max->y;
    points[2].z = min->z;

    points[3].x = min->x;
    points[3].y = max->y;
    points[3].z = min->z;

	// points for Face2 in XY
    points[4].x = min->x;
    points[4].y = min->y;
    points[4].z = max->z;

    points[5].x = max->x;
    points[5].y = min->y;
    points[5].z = max->z;

    points[6].x = max->x;
    points[6].y = max->y;
    points[6].z = max->z;

    points[7].x = min->x;
    points[7].y = max->y;
    points[7].z = max->z;


	// points for Face1 in YZ
    points[8].x = min->x;
    points[8].y = min->y;
    points[8].z = min->z;

    points[9].x = min->x;
    points[9].y = max->y;
    points[9].z = min->z;
    
    points[10].x = min->x;
    points[10].y = max->y;
    points[10].z = max->z;

    points[11].x = min->x;
    points[11].y = min->y;
    points[11].z = max->z;

	// points for Face2 in YZ
    points[12].x = max->x;
    points[12].y = min->y;
    points[12].z = min->z;

    points[13].x = max->x;
    points[13].y = max->y;
    points[13].z = min->z;
    
    points[14].x = max->x;
    points[14].y = max->y;
    points[14].z = max->z;

    points[15].x = max->x;
    points[15].y = min->y;
    points[15].z = max->z;

	// points for Face1 in XZ
    points[16].x = min->x;
    points[16].y = min->y;
    points[16].z = min->z;

    points[17].x = max->x;
    points[17].y = min->y;
    points[17].z = min->z;
    
    points[18].x = max->x;
    points[18].y = min->y;
    points[18].z = max->z;

    points[19].x = min->x;
    points[19].y = min->y;
    points[19].z = max->z;

	// points for Face2 in XZ
    points[20].x = min->x;
    points[20].y = max->y;
    points[20].z = min->z;

    points[21].x = max->x;
    points[21].y = max->y;
    points[21].z = min->z;
    
    points[22].x = max->x;
    points[22].y = max->y;
    points[22].z = max->z;

    points[23].x = min->x;
    points[23].y = max->y;
    points[23].z = max->z;


}


